# Filter units ----------------------------------------------------------------------------------------------------
# Filter out units that have "Not computed" firing pattern
filter_units <- function(Data) {
  tryCatch({

    flog.debug("Filtering out units with: Not computed", name = log.name)
    Filtered.data <- Data %>%
      mutate(Sorted = if_else(str_detect(Unit, pattern = "SPK..i"), "Unsorted", "Sorted")) %>%
      select_at(c(group.columns, "Unit", "Event", "Direction","Sorted")) %>% 
      unique() %>%
      spread(key = Event, value = Direction)
    
    Filtered.data[is.na(Filtered.data)] <- "Not available"
    
    Filtered.data <- Filtered.data %>%
      filter_all(all_vars(. != "Not computed")) %>% 
      gather(key = "Event", value = "Direction", one_of(perievent.columns))
    
    Filtered.data <- left_join(Filtered.data, Data, by = c(group.columns, "Unit", "Event", "Direction"))
    
    Filtered.data <- Filtered.data %>% filter(Sorted == "Sorted") %>%
      unite(UnitID, File.name, SessionID, Unit) %>% select(-FileID, -Sorted)
    
    Data <- NULL
    
    return(Filtered.data)
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch
}


# Calculate mean ----------------------------------------------------------------------------------------------------
# Calculate the mean firing rate for each Direction or as population vector and merge with the one.sample t.test 
# permutation calculations. 
# If use with Direction, the group.cols = c("Area.brain", "TreatmentID", "Event", "norm.time", "Direction)

calculate_mean <- function(Data, permuted.stats.data = NULL, 
                           group.cols = c("Area.brain", "TreatmentID", "Event", "norm.time")) {
  tryCatch({
    
    Data.mean <- Data  %>%
      group_by_at(group.cols) %>%
      summarise(avg = mean(z.score), N = n(), sd = sd(z.score), sem = sd/sqrt(N)) %>%
      ungroup() %>% 
      rename(z.score = avg)
    
    
    if (is_empty(permuted.stats.data)) {
      return(Data.mean)
    } else {
      
      permuted.stats.data <- permuted.stats.data %>%
        rename(norm.time  = time) 
      
      Data.mean <- Data.mean %>%
        left_join(., permuted.stats.data, 
                by = c(group.cols))
      return(Data.mean)
    }
      
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch
}


# Calculate distribution ----------------------------------------------------------------------------------------------------
# Calculate the distribution of the units as of Direction

calculate_distribution <- function(Data) {
  tryCatch({
    
    if (!("UnitID" %in% colnames(Data))) {
      
      Data <- Data %>% unite(UnitID, File.name, SessionID, Unit)
      
    }
      
    UnitNo <- Data %>%
      select(Area.brain, TreatmentID, Direction, Event, UnitID) %>%
      unique() %>% 
      group_by(Area.brain, TreatmentID, Direction, Event) %>% 
      summarise(n = n()) %>% ungroup() %>%
      group_by(Area.brain, TreatmentID, Event) %>%
      mutate(perc = (n/sum(n))*100) %>% ungroup()
    
    return(UnitNo)
 
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch
}